SET client_min_messages TO warning;
DROP SCHEMA IF EXISTS mv_benchmark CASCADE;
CREATE SCHEMA mv_benchmark;
SET search_path TO mv_benchmark, public;

-- ============================================================================
-- BASE TABLES (Integer IDs)
-- ============================================================================
CREATE TABLE invoices (
                          invoice_id int GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
                          invoice_num text NOT NULL
);

CREATE TABLE invoice_lines (
                               line_id int GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY,
                               invoice_id int NOT NULL REFERENCES invoices(invoice_id),
                               amount numeric(10,2) NOT NULL,
                               qty int NOT NULL
);
CREATE INDEX ON invoice_lines(invoice_id);

-- ============================================================================
-- MATERIALIZED VIEW
-- ============================================================================
CREATE MATERIALIZED VIEW invoice_summary AS
SELECT
    i.invoice_id,
    i.invoice_num,
    COUNT(l.line_id) as line_count,
    COALESCE(SUM(l.amount * l.qty), 0) as total_amount
FROM invoices i
         LEFT JOIN invoice_lines l USING (invoice_id)
GROUP BY i.invoice_id;

CREATE UNIQUE INDEX ON invoice_summary(invoice_id);

-- ============================================================================
-- MATERIALIZED TABLE (Control Group)
-- ============================================================================
CREATE TABLE invoice_summary_table (
                                       invoice_id int PRIMARY KEY,
                                       invoice_num text NOT NULL,
                                       line_count bigint NOT NULL DEFAULT 0,
                                       total_amount numeric(10,2) NOT NULL DEFAULT 0
);

-- ============================================================================
-- FUNCTIONS & TRIGGERS
-- ============================================================================

-- 1. Manual Table Maintenance (The "Old Way")
CREATE OR REPLACE FUNCTION update_summary_table() RETURNS TRIGGER AS $$
DECLARE
    affected_ids int[];
BEGIN
    IF TG_OP = 'INSERT' THEN
        SELECT array_agg(DISTINCT invoice_id) INTO affected_ids FROM new_table;
    ELSIF TG_OP = 'UPDATE' THEN
        SELECT array_agg(DISTINCT invoice_id) INTO affected_ids FROM (SELECT invoice_id FROM new_table UNION SELECT invoice_id FROM old_table) t;
    ELSIF TG_OP = 'DELETE' THEN
        SELECT array_agg(DISTINCT invoice_id) INTO affected_ids FROM old_table;
    END IF;

    IF affected_ids IS NOT NULL THEN
        -- Delete orphaned rows
        DELETE FROM mv_benchmark.invoice_summary_table
        WHERE invoice_id = ANY(affected_ids);

        -- Upsert current rows
        INSERT INTO mv_benchmark.invoice_summary_table (invoice_id, invoice_num, line_count, total_amount)
        SELECT
            i.invoice_id,
            i.invoice_num,
            COUNT(l.line_id) as line_count,
            COALESCE(SUM(l.amount * l.qty), 0) as total_amount
        FROM mv_benchmark.invoices i
                 LEFT JOIN mv_benchmark.invoice_lines l USING (invoice_id)
        WHERE i.invoice_id = ANY(affected_ids)
        GROUP BY i.invoice_id
        ON CONFLICT (invoice_id) DO UPDATE SET
        line_count = EXCLUDED.line_count,
        total_amount = EXCLUDED.total_amount;
    END IF;
    RETURN NULL;
END;
$$ LANGUAGE plpgsql;

-- 2. Partial Refresh Trigger
CREATE OR REPLACE FUNCTION partial_refresh_mv() RETURNS TRIGGER AS $$
DECLARE
    affected_ids int[];
BEGIN
    IF TG_OP = 'INSERT' THEN
        SELECT array_agg(DISTINCT invoice_id) INTO affected_ids FROM new_table;
    ELSIF TG_OP = 'UPDATE' THEN
        SELECT array_agg(DISTINCT invoice_id) INTO affected_ids FROM (SELECT invoice_id FROM new_table UNION SELECT invoice_id FROM old_table) t;
    ELSIF TG_OP = 'DELETE' THEN
        SELECT array_agg(DISTINCT invoice_id) INTO affected_ids FROM old_table;
    END IF;

    IF affected_ids IS NOT NULL THEN
        -- Sort IDs to minimize deadlock risk during high concurrency
        SELECT array_agg(id ORDER BY id) INTO affected_ids FROM unnest(affected_ids) AS id;

        EXECUTE 'REFRESH MATERIALIZED VIEW mv_benchmark.invoice_summary WHERE invoice_id = ANY($1)'
            USING affected_ids;
    END IF;
    RETURN NULL;
END;
$$ LANGUAGE plpgsql;

-- Trigger Toggles
CREATE OR REPLACE FUNCTION enable_materialized_table_triggers() RETURNS void AS $$
BEGIN
    PERFORM mv_benchmark.disable_all_triggers();
    CREATE TRIGGER t_mat_table_ins AFTER INSERT ON mv_benchmark.invoice_lines REFERENCING NEW TABLE AS new_table FOR EACH STATEMENT EXECUTE FUNCTION mv_benchmark.update_summary_table();
    CREATE TRIGGER t_mat_table_upd AFTER UPDATE ON mv_benchmark.invoice_lines REFERENCING NEW TABLE AS new_table OLD TABLE AS old_table FOR EACH STATEMENT EXECUTE FUNCTION mv_benchmark.update_summary_table();
    CREATE TRIGGER t_mat_table_del AFTER DELETE ON mv_benchmark.invoice_lines REFERENCING OLD TABLE AS old_table FOR EACH STATEMENT EXECUTE FUNCTION mv_benchmark.update_summary_table();
END; $$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION enable_partial_refresh_triggers() RETURNS void AS $$
BEGIN
    PERFORM mv_benchmark.disable_all_triggers();
    CREATE TRIGGER t_partial_ins AFTER INSERT ON mv_benchmark.invoice_lines REFERENCING NEW TABLE AS new_table FOR EACH STATEMENT EXECUTE FUNCTION mv_benchmark.partial_refresh_mv();
    CREATE TRIGGER t_partial_upd AFTER UPDATE ON mv_benchmark.invoice_lines REFERENCING NEW TABLE AS new_table OLD TABLE AS old_table FOR EACH STATEMENT EXECUTE FUNCTION mv_benchmark.partial_refresh_mv();
    CREATE TRIGGER t_partial_del AFTER DELETE ON mv_benchmark.invoice_lines REFERENCING OLD TABLE AS old_table FOR EACH STATEMENT EXECUTE FUNCTION mv_benchmark.partial_refresh_mv();
END; $$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION disable_all_triggers() RETURNS void AS $$
BEGIN
    DROP TRIGGER IF EXISTS t_mat_table_ins ON mv_benchmark.invoice_lines;
    DROP TRIGGER IF EXISTS t_mat_table_upd ON mv_benchmark.invoice_lines;
    DROP TRIGGER IF EXISTS t_mat_table_del ON mv_benchmark.invoice_lines;
    DROP TRIGGER IF EXISTS t_partial_ins ON mv_benchmark.invoice_lines;
    DROP TRIGGER IF EXISTS t_partial_upd ON mv_benchmark.invoice_lines;
    DROP TRIGGER IF EXISTS t_partial_del ON mv_benchmark.invoice_lines;
END; $$ LANGUAGE plpgsql;

-- Data Generator
CREATE OR REPLACE FUNCTION generate_data(p_count int) RETURNS void AS $$
BEGIN
    RAISE NOTICE 'Generating % invoices with ~5 lines each...', p_count;

    -- Clear existing data
    TRUNCATE mv_benchmark.invoices, mv_benchmark.invoice_lines CASCADE;
    ALTER TABLE mv_benchmark.invoices ALTER COLUMN invoice_id RESTART WITH 1;
    ALTER TABLE mv_benchmark.invoice_lines ALTER COLUMN line_id RESTART WITH 1;

    INSERT INTO mv_benchmark.invoices (invoice_num)
    SELECT 'INV-' || to_char(i, 'FM000000') FROM generate_series(1, p_count) i;

    INSERT INTO mv_benchmark.invoice_lines (invoice_id, amount, qty)
    SELECT invoice_id, (random()*100)::numeric(10,2), (random()*5+1)::int
    FROM mv_benchmark.invoices CROSS JOIN generate_series(1, 5);

    REFRESH MATERIALIZED VIEW mv_benchmark.invoice_summary;

    TRUNCATE mv_benchmark.invoice_summary_table;
    INSERT INTO mv_benchmark.invoice_summary_table SELECT * FROM mv_benchmark.invoice_summary;

    -- Analyze tables
    ANALYZE mv_benchmark.invoices;
    ANALYZE mv_benchmark.invoice_lines;
    ANALYZE mv_benchmark.invoice_summary;
    ANALYZE mv_benchmark.invoice_summary_table;

    RAISE NOTICE 'Data generation complete. Rows: invoices=%, lines=%',
        p_count, p_count * 5;
END; $$ LANGUAGE plpgsql;

-- ============================================================================
-- VERIFICATION FUNCTION
-- ============================================================================
CREATE OR REPLACE FUNCTION verify_consistency(check_target text DEFAULT 'both')
    RETURNS TABLE(
                     check_name text,
                     status text,
                     details text
                 ) AS $$
BEGIN
    -- Only check MV if requested
    IF check_target IN ('both', 'mv') THEN
        -- MV: Data Mismatches check
        RETURN QUERY
            SELECT
                'MV: Data Mismatches'::text,
                CASE WHEN COUNT(*) = 0 THEN 'PASS' ELSE 'FAIL' END::text,
                CASE
                    WHEN COUNT(*) = 0 THEN 'OK'
                    ELSE 'Mismatches: ' || COUNT(*)::text
                    END::text
            FROM (
                     SELECT i.invoice_id
                     FROM mv_benchmark.invoices i
                              LEFT JOIN mv_benchmark.invoice_summary mv USING (invoice_id)
                              LEFT JOIN LATERAL (
                         SELECT
                             COUNT(l.line_id) as actual_count,
                             COALESCE(SUM(l.amount * l.qty), 0) as actual_amount
                         FROM mv_benchmark.invoice_lines l
                         WHERE l.invoice_id = i.invoice_id
                         ) actual ON true
                     WHERE
                         mv.invoice_id IS NULL
                        OR mv.invoice_num != i.invoice_num
                        OR mv.line_count != actual.actual_count
                        OR ABS(mv.total_amount - actual.actual_amount) > 0.01
                 ) mismatches;

        -- MV: Orphaned Rows check
        RETURN QUERY
            SELECT
                'MV: Orphaned Rows'::text,
                CASE WHEN COUNT(*) = 0 THEN 'PASS' ELSE 'FAIL' END::text,
                CASE
                    WHEN COUNT(*) = 0 THEN 'OK'
                    ELSE 'Orphans: ' || COUNT(*)::text
                    END::text
            FROM (
                     SELECT mv.invoice_id
                     FROM mv_benchmark.invoice_summary mv
                              LEFT JOIN mv_benchmark.invoices i USING (invoice_id)
                     WHERE i.invoice_id IS NULL
                 ) orphans;
    END IF;

    -- Only check table if requested
    IF check_target IN ('both', 'table') THEN
        -- Mat Table: Data Mismatches check
        RETURN QUERY
            SELECT
                'Mat Table: Data Mismatches'::text,
                CASE WHEN COUNT(*) = 0 THEN 'PASS' ELSE 'FAIL' END::text,
                CASE
                    WHEN COUNT(*) = 0 THEN 'OK'
                    ELSE 'Mismatches: ' || COUNT(*)::text
                    END::text
            FROM (
                     SELECT i.invoice_id
                     FROM mv_benchmark.invoices i
                              LEFT JOIN mv_benchmark.invoice_summary_table mt USING (invoice_id)
                              LEFT JOIN LATERAL (
                         SELECT
                             COUNT(l.line_id) as actual_count,
                             COALESCE(SUM(l.amount * l.qty), 0) as actual_amount
                         FROM mv_benchmark.invoice_lines l
                         WHERE l.invoice_id = i.invoice_id
                         ) actual ON true
                     WHERE
                         mt.invoice_id IS NULL
                        OR mt.invoice_num != i.invoice_num
                        OR mt.line_count != actual.actual_count
                        OR ABS(mt.total_amount - actual.actual_amount) > 0.01
                 ) mismatches;

        -- Mat Table: Orphaned Rows check
        RETURN QUERY
            SELECT
                'Mat Table: Orphaned Rows'::text,
                CASE WHEN COUNT(*) = 0 THEN 'PASS' ELSE 'FAIL' END::text,
                CASE
                    WHEN COUNT(*) = 0 THEN 'OK'
                    ELSE 'Orphans: ' || COUNT(*)::text
                    END::text
            FROM (
                     SELECT mt.invoice_id
                     FROM mv_benchmark.invoice_summary_table mt
                              LEFT JOIN mv_benchmark.invoices i USING (invoice_id)
                     WHERE i.invoice_id IS NULL
                 ) orphans;
    END IF;

    -- Row count summary
    RETURN QUERY
        SELECT
            'Row Counts'::text,
            'INFO'::text,
            format('Base: %s invoices, %s lines | MV: %s rows | Mat Table: %s rows',
                   (SELECT COUNT(*) FROM mv_benchmark.invoices),
                   (SELECT COUNT(*) FROM mv_benchmark.invoice_lines),
                   (SELECT COUNT(*) FROM mv_benchmark.invoice_summary),
                   (SELECT COUNT(*) FROM mv_benchmark.invoice_summary_table)
            )::text;
END;
$$ LANGUAGE plpgsql;

-- Print setup summary
DO $$
    BEGIN
        RAISE NOTICE '============================================================';
        RAISE NOTICE 'Materialized View Benchmark Schema Created';
        RAISE NOTICE '============================================================';
        RAISE NOTICE 'Tables: invoices, invoice_lines';
        RAISE NOTICE 'Materialized View: invoice_summary';
        RAISE NOTICE 'Materialized Table: invoice_summary_table';
        RAISE NOTICE '';
        RAISE NOTICE 'Usage:';
        RAISE NOTICE '  1. Generate data: SELECT generate_data(10000);';
        RAISE NOTICE '  2. Enable triggers:';
        RAISE NOTICE '     - Materialized table: SELECT enable_materialized_table_triggers();';
        RAISE NOTICE '     - Partial refresh: SELECT enable_partial_refresh_triggers();';
        RAISE NOTICE '     - Disable all: SELECT disable_all_triggers();';
        RAISE NOTICE '  3. Verify: SELECT * FROM verify_consistency();';
        RAISE NOTICE '============================================================';
    END $$;
